﻿namespace Microsoft.Samples.PlanMyNight.Data
{
    using System;
    using System.Collections.Generic;
    using System.Globalization;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Microsoft.Samples.PlanMyNight.Bing;
    using Microsoft.Samples.PlanMyNight.Entities;

    public class BingActivitiesRepository : IActivitiesRepository
    {
        private const string ActivityKeyPattern = "{0}|{1}|{2}~{3}";

        private readonly TimeSpan TokenCacheExpiration = TimeSpan.FromMinutes(180);

        private readonly IBingMapsService service;

        private static BingToken cachedToken;

        public BingActivitiesRepository() :
            this(new BingMapsService())
        {
        }

        public BingActivitiesRepository(IBingMapsService service)
        {
            this.service = service;
        }

        public static void InvalidateTokenCache()
        {
            cachedToken = null;
        }

        public PagingResult<Activity> Search(AdvancedSearchQuery searchCriteria)
        {
            if (searchCriteria.Radius.HasValue && searchCriteria.Radius.Value > 0)
            {
                searchCriteria.SortBy = SortCriteria.Distance;
            }

            var token = this.RetrieveBingToken();
            var result = this.service.SearchActivities(searchCriteria, token);

            if (searchCriteria.Radius.HasValue && searchCriteria.Radius.Value > 0)
            {
                var itemsOutOfRange = result.Items.Where(a => a.Distance > searchCriteria.Radius.Value).ToArray();
                if (itemsOutOfRange.Length > 0)
                {
                    // items out of range, remove and update paging information
                    result = new PagingResult<Activity>(result.Items.Where(a => !itemsOutOfRange.Select(o => o.BingId).Contains(a.BingId)).ToArray())
                    {
                        CurrentPage = result.CurrentPage,
                        PageSize = result.PageSize,
                        TotalItems = (result.PageSize * (result.CurrentPage - 1)) + result.PageSize - itemsOutOfRange.Length
                    };
                }
            }

            foreach (var item in result.Items)
            {
                UpdateActivityId(item, searchCriteria.ActivityTypeId);
            }

            return result;
        }

        public PagingResult<Activity> Search(NaturalSearchQuery query)
        {
            var token = this.RetrieveBingToken();
            var result = this.service.SearchActivities(query, token);

            foreach (var item in result.Item1.Items)
            {
                UpdateActivityId(item, query.ActivityTypeId);
            }

            return result.Item1;
        }

        public Tuple<double, double> GeocodeAddress(ActivityAddress address)
        {
            var token = this.RetrieveBingToken();
            var location = this.service.GeocodeAddress(address, token);
            var result = new Tuple<double, double>(location.Longitude, location.Latitude);

            return result;
        }

        public ActivityAddress ParseQueryLocation(string query)
        {
            var token = this.RetrieveBingToken();
            var anyActivityId = this.RetrieveActivityTypes().ToArray()[0].Id;
            var result = this.service.SearchActivities(new NaturalSearchQuery { Query = query, ActivityTypeId = anyActivityId, PageSize = 1, Page = 1, SortBy = SortCriteria.Relevance, Type = SearchType.Activity }, token);

            return result.Item2;
        }

        public Activity RetrieveActivity(string id)
        {
            var rawKey = Encoding.Default.GetString(Convert.FromBase64String(id)).Split('|');
            var bingId = rawKey[0];
            var typeId = int.Parse(rawKey[1], CultureInfo.InvariantCulture);

            var rawLocation = rawKey[2].Split('~');
            var longitude = double.Parse(rawLocation[0], CultureInfo.InvariantCulture);
            var latitude = double.Parse(rawLocation[1], CultureInfo.InvariantCulture);

            var searchCriteria = new AdvancedSearchQuery
            {
                ActivityTypeId = typeId,
                Longitude = longitude,
                Latitude = latitude,
                SortBy = SortCriteria.Distance,
                PageSize = 10,
                Page = 1,
            };

            var token = this.RetrieveBingToken();
            int retry = 0;
            Activity item = null;
            while (item == null)
            {
                var result = this.service.SearchActivities(searchCriteria, token);
                item = result.Items.Where(i => i.BingId == bingId).SingleOrDefault();
                searchCriteria.Page++;
                retry++;
                if (retry == 3) break;
            }

            if (item != null)
            {
                UpdateActivityId(item, typeId);
            }

            return item;
        }

        public void PopulateItineraryActivities(Itinerary itinerary)
        {
            // Sequencial retrieval
            ////foreach (var item in itinerary.Activities.Where(i => i.Activity == null))
            ////{
            ////    item.Activity = this.RetrieveActivity(item.ActivityId);
            ////}

            // Parallel retrieval
            Parallel.ForEach(itinerary.Activities.Where(i => i.Activity == null),
                item =>
                {
                    item.Activity = this.RetrieveActivity(item.ActivityId);
                });
        }

        public IEnumerable<ActivityType> RetrieveActivityTypes()
        {
            // Mocked data
            return new[]
            {
                new ActivityType { Id = 11168, Name = "Restaurant", PluralizedName = "Restaurants" },
                new ActivityType { Id = 11199, Name = "Bar", PluralizedName = "Bars" },
                new ActivityType { Id = 10028, Name = "Club", PluralizedName = "Clubs" }
            };
        }

        private static void UpdateActivityId(Activity item, int typeId)
        {
            var rawKey = string.Format(CultureInfo.InvariantCulture, ActivityKeyPattern, item.BingId, typeId, item.Location[0], item.Location[1]);
            item.Id = Convert.ToBase64String(Encoding.Default.GetBytes(rawKey));
            item.TypeId = typeId;
        }

        private string RetrieveBingToken()
        {
            if (cachedToken != null && !cachedToken.Expired)
            {
                return cachedToken.Value;
            }

            var token = this.service.GetClientToken(this.RetrieveIpAddress(), (int)this.TokenCacheExpiration.TotalMinutes + 10);
            cachedToken = new BingToken(this.TokenCacheExpiration, token);
            return token;
        }

        private string RetrieveIpAddress()
        {
            // TODO: return server ip address
            return "127.0.0.1";
        }
    }

    public class BingToken
    {
        public BingToken(TimeSpan expirationFromNow, string token)
        {
            this.Expiration = DateTime.UtcNow.AddMinutes(expirationFromNow.TotalMinutes);
            this.Value = token;
        }

        public bool Expired
        {
            get { return DateTime.UtcNow > this.Expiration; }
        }

        public DateTime Expiration { get; private set; }

        public string Value { get; private set; }
    }
}
